// Investment Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
    initInvestmentPage();
});

function initInvestmentPage() {
    initProfitChart();         // MUST BE FIRST
    initInvestmentTabs();
    initCalculator();          // AFTER CHART
    initInvestmentForm();
    initPlanCards();
    initInvestmentModal();
}

/* ---------------------------
   INVESTMENT TABS
---------------------------- */
function initInvestmentTabs() {
    const tabs = document.querySelectorAll('.tab-btn');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const type = this.dataset.tab;

            tabs.forEach(t => t.classList.remove('active'));
            this.classList.add('active');
            filterInvestments(type);
        });
    });
}

function filterInvestments(type) {
    const items = document.querySelectorAll('.investment-item');
    
    items.forEach(item => {
        if (type === 'all') item.style.display = 'flex';
        else if (type === 'active') item.style.display = 'flex';
        else if (type === 'completed') item.style.display = 'none';
    });
}

/* ---------------------------
   GLOBAL INVESTMENT SUMMARY FUNCTION
---------------------------- */
function updateInvestmentSummary(plan, amount) {
    const planData = {
        bronze: { name: 'Bronze Plan', daily: 1, duration: 30, total: '30%' },
        silver: { name: 'Silver Plan', daily: 1.2, duration: 180, total: '72%' },
        gold: { name: 'Gold Plan', daily: 1.5, duration: 365, total: '108%' },
        platinum: { name: 'Platinum Plan', daily: 2, duration: 730, total: '146%' }
    };

    const p = planData[plan];
    const dailyProfit = amount * (p.daily / 100);
    const totalProfit = dailyProfit * p.duration;

    document.getElementById('summary-plan').textContent = p.name;
    document.getElementById('summary-daily').textContent = p.daily + "% Daily";
    document.getElementById('summary-duration').textContent = p.duration + " Days";
    document.getElementById('summary-total').textContent = p.total;
    document.getElementById('summary-profit').textContent = `$${totalProfit.toFixed(2)}`;
}


/* ---------------------------
   CALCULATOR SYSTEM
---------------------------- */
function initCalculator() {
    const planSelect = document.getElementById('calc-plan');
    const amountInput = document.getElementById('calc-amount');

    const durationText = document.getElementById('calc-duration-text');
    const durationMonths = document.getElementById('calc-duration-months');

    const plans = {
        bronze: { dailyReturn: 0.01, duration: 30, months: "1 Month" },
        silver: { dailyReturn: 0.012, duration: 180, months: "6 Months" },
        gold: { dailyReturn: 0.015, duration: 365, months: "12 Months" },
        platinum: { dailyReturn: 0.02, duration: 730, months: "24 Months" }
    };

    function updateCalculator() {
        const plan = planSelect.value;
        const amount = parseFloat(amountInput.value) || 1000;
        const p = plans[plan];

        durationText.textContent = p.duration + " Days";
        durationMonths.textContent = "(" + p.months + ")";

        const daily = amount * p.dailyReturn;
        const weekly = daily * 7;
        const monthly = daily * 30;
        const totalProfit = daily * p.duration;
        const totalReturn = amount + totalProfit;

        document.getElementById('daily-profit').textContent = `$${daily.toFixed(2)}`;
        document.getElementById('weekly-profit').textContent = `$${weekly.toFixed(2)}`;
        document.getElementById('monthly-profit').textContent = `$${monthly.toFixed(2)}`;
        document.getElementById('total-profit').textContent = `$${totalProfit.toFixed(2)}`;
        document.getElementById('total-return').textContent = `$${totalReturn.toFixed(2)}`;

        updateProfitChart(plan, amount);
        updateInvestmentSummary(plan, amount);
    }

    planSelect.addEventListener('change', updateCalculator);
    amountInput.addEventListener('input', updateCalculator);

    updateCalculator(); // Initial
}

/* ---------------------------
   PROFIT CHART
---------------------------- */

let profitChart = null;

function initProfitChart() {
    const canvas = document.getElementById('profit-chart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');

    profitChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: [],
            datasets: [{
                label: 'Cumulative Profit',
                data: [],
                borderColor: 'rgba(87,193,232,1)',
                backgroundColor: 'rgba(87,193,232,0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { display: false } },
            scales: {
                y: { beginAtZero: true },
                x: {}
            }
        }
    });
}

function updateProfitChart(plan, amount) {
    if (!profitChart) return;

    const plans = {
        bronze: { dailyReturn: 0.01, duration: 30 },
        silver: { dailyReturn: 0.012, duration: 180 },
        gold: { dailyReturn: 0.015, duration: 365 },
        platinum: { dailyReturn: 0.02, duration: 730 }
    };

    const p = plans[plan];
    const months = Math.min(6, p.duration / 30);

    const labels = [];
    const profits = [];

    for (let i = 1; i <= months; i++) {
        labels.push("Month " + i);
        profits.push(amount * p.dailyReturn * 30 * i);
    }

    profitChart.data.labels = labels;
    profitChart.data.datasets[0].data = profits;
    profitChart.update();
}

/* ---------------------------
   INVESTMENT FORM
---------------------------- */
function initInvestmentForm() {
    const form = document.getElementById('investment-form');
    const planSelect = document.getElementById('investment-plan');
    const amountInput = document.getElementById('investment-amount-input');
    const presets = document.querySelectorAll('.amount-preset');

    const planData = {
        bronze: { name: 'Bronze Plan', daily: 1, duration: 30, total: '30%' },
        silver: { name: 'Silver Plan', daily: 1.2, duration: 180, total: '72%' },
        gold: { name: 'Gold Plan', daily: 1.5, duration: 365, total: '108%' },
        platinum: { name: 'Platinum Plan', daily: 2, duration: 730, total: '146%' }
    };

/* ---------------------------
   GLOBAL INVESTMENT SUMMARY FUNCTION
---------------------------- */
    presets.forEach(btn => {
        btn.addEventListener('click', function() {
            presets.forEach(p => p.classList.remove('active'));
            this.classList.add('active');

            amountInput.value = this.dataset.amount;
            updateInvestmentSummary(planSelect.value, parseFloat(this.dataset.amount));
        });
    });

    planSelect.addEventListener('change', () => {
        updateInvestmentSummary(planSelect.value, parseFloat(amountInput.value) || 100);
    });
    amountInput.addEventListener('input', () => {
        presets.forEach(p => p.classList.remove('active'));
        updateInvestmentSummary(planSelect.value, parseFloat(amountInput.value) || 100);
    });

    form.addEventListener('submit', e => {
        e.preventDefault();
        const plan = planSelect.value;
        const amount = parseFloat(amountInput.value);

        if (!plan || amount < 100) {
            showError("Invalid investment.");
            return;
        }

        showInvestmentConfirmation(plan, amount);
    });

    updateInvestmentSummary("gold", 1000);
}

/* ---------------------------
   PLAN CARDS
---------------------------- */
function initPlanCards() {
    const buttons = document.querySelectorAll('.invest-btn');

    buttons.forEach(btn => {
        btn.addEventListener('click', function() {
            const plan = this.dataset.plan;

            const defaults = { bronze:100, silver:500, gold:1000, platinum:5000 };
            const amount = defaults[plan];

            document.getElementById('investment-plan').value = plan;
            document.getElementById('investment-amount-input').value = amount;

            updateInvestmentSummary(plan, amount);

            document.querySelector('.investment-form').scrollIntoView({ behavior: 'smooth' });
        });
    });
}

/* ---------------------------
   MODAL + PROCESS
---------------------------- */
function showInvestmentConfirmation(plan, amount) {
    const modal = document.getElementById('investment-modal');
    const content = modal.querySelector('.investment-confirmation');

    const data = {
        bronze: { name: 'Bronze Plan', daily: 1, duration: 30 },
        silver: { name: 'Silver Plan', daily: 1.2, duration: 180 },
        gold: { name: 'Gold Plan', daily: 1.5, duration: 365 },
        platinum: { name: 'Platinum Plan', daily: 2, duration: 730 }
    };

    const p = data[plan];
    const dailyProfit = amount * (p.daily / 100);
    const totalProfit = dailyProfit * p.duration;

    content.innerHTML = `
        <div class="confirmation-details">
            <div class="detail-item"><span>Plan:</span><span>${p.name}</span></div>
            <div class="detail-item"><span>Amount:</span><span>$${amount}</span></div>
            <div class="detail-item"><span>Daily Profit:</span><span>$${dailyProfit.toFixed(2)}</span></div>
            <div class="detail-item total"><span>Total Profit:</span><span>$${totalProfit.toFixed(2)}</span></div>
        </div>
        <div class="modal-actions">
            <button class="btn-secondary close-modal">Cancel</button>
            <button class="btn-primary" id="confirm-investment-btn">Confirm</button>
        </div>
    `;

    modal.style.display = "flex";

    modal.querySelector('.close-modal').addEventListener('click', () => modal.style.display = "none");
    modal.querySelector('#confirm-investment-btn').addEventListener('click', () => {
        processInvestment(plan, amount);
        modal.style.display = "none";
    });
}

function processInvestment(plan, amount) {
    console.log("Investment processed →", plan, amount);
}

/* ---------------------------
   MODAL HANDLING
---------------------------- */
function initInvestmentModal() {
    const modal = document.getElementById('investment-modal');
    if (!modal) return;

    document.addEventListener('keydown', e => {
        if (e.key === "Escape") modal.style.display = "none";
    });
}

/* ---------------------------
   UTIL
---------------------------- */
function showError(msg) {
    alert(msg);
}

/* Highlight animation */
const style = document.createElement("style");
style.textContent = `
@keyframes highlight {
    0%{box-shadow:0 0 0 0 rgba(255,215,0,.8)}
    70%{box-shadow:0 0 0 12px rgba(255,215,0,0)}
    100%{box-shadow:0 0 0 0 rgba(255,215,0,0)}
}`;
document.head.appendChild(style);
