// Login Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize common functions
    if (window.CommonFunctions) {
        window.CommonFunctions.initPasswordToggle();
        window.CommonFunctions.initTooltips();
    }
    
    // Initialize login form
    initLoginForm();
    
    // Initialize social login buttons
    initSocialLogin();
    
    // Initialize forgot password
    initForgotPassword();
});

// Initialize login form
function initLoginForm() {
    const loginForm = document.getElementById('loginForm');
    
    if (!loginForm) return;
    
    loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Clear previous validation messages
        if (window.CommonFunctions) {
            window.CommonFunctions.clearValidationMessages();
        }
        
        // Get form values
        const email = document.getElementById('login-email').value.trim();
        const password = document.getElementById('login-password').value;
        const rememberMe = document.getElementById('remember-me').checked;
        
        // Validate inputs
        let isValid = true;
        
        // Email validation
        if (!email) {
            showFieldError('login-email', 'Email is required');
            isValid = false;
        } else if (!isValidEmail(email)) {
            showFieldError('login-email', 'Please enter a valid email address');
            isValid = false;
        }
        
        // Password validation
        if (!password) {
            showFieldError('login-password', 'Password is required');
            isValid = false;
        } else if (password.length < 6) {
            showFieldError('login-password', 'Password must be at least 6 characters');
            isValid = false;
        }
        
        if (!isValid) return;
        
        // Simulate login process
        simulateLogin(email, password, rememberMe);
    });
    
    // Real-time validation
    const emailInput = document.getElementById('login-email');
    const passwordInput = document.getElementById('login-password');
    
    if (emailInput) {
        emailInput.addEventListener('blur', function() {
            const email = this.value.trim();
            if (email && !isValidEmail(email)) {
                showFieldError('login-email', 'Please enter a valid email address');
            } else {
                clearFieldError('login-email');
            }
        });
    }
    
    if (passwordInput) {
        passwordInput.addEventListener('blur', function() {
            const password = this.value;
            if (password && password.length < 6) {
                showFieldError('login-password', 'Password must be at least 6 characters');
            } else {
                clearFieldError('login-password');
            }
        });
    }
}

// Initialize social login buttons
function initSocialLogin() {
    const googleBtn = document.querySelector('.google-btn');
    const microsoftBtn = document.querySelector('.microsoft-btn');
    
    if (googleBtn) {
        googleBtn.addEventListener('click', function() {
            // In a real app, this would redirect to Google OAuth
            console.log('Google login clicked');
            showSocialLoginMessage('Google');
        });
    }
    
    if (microsoftBtn) {
        microsoftBtn.addEventListener('click', function() {
            // In a real app, this would redirect to Microsoft OAuth
            console.log('Microsoft login clicked');
            showSocialLoginMessage('Microsoft');
        });
    }
}

// Initialize forgot password
function initForgotPassword() {
    const forgotPasswordLink = document.querySelector('.forgot-password');
    
    if (forgotPasswordLink) {
        forgotPasswordLink.addEventListener('click', function(e) {
            e.preventDefault();
            showForgotPasswordModal();
        });
    }
}

// Simulate login process
function simulateLogin(email, password, rememberMe) {
    const submitBtn = document.querySelector('#loginForm .submit-btn');
    
    // Show loading state
    if (window.CommonFunctions) {
        window.CommonFunctions.setButtonLoading(submitBtn, true);
    }
    
    // Simulate API call delay
    setTimeout(() => {
        // In a real application, this would be an API call to your backend
        console.log('Login attempt:', { email, password, rememberMe });
        
        // For demo purposes, simulate successful login
        const isSuccess = Math.random() > 0.2; // 80% success rate for demo
        
        if (isSuccess) {
            // Show success message
            if (window.CommonFunctions) {
                window.CommonFunctions.showSuccessNotification('Login successful! Redirecting to dashboard...');
            }
            
            // Redirect to dashboard after delay
            setTimeout(() => {
                // In a real application, redirect to dashboard
                // window.location.href = 'dashboard.html';
                console.log('Redirecting to dashboard...');
                
                // For demo, show alert
                alert('Login successful! In a real application, you would be redirected to your investment dashboard.');
            }, 1500);
        } else {
            // Show error message
            if (window.CommonFunctions) {
                window.CommonFunctions.showErrorNotification('Invalid email or password. Please try again.');
            }
            
            // Show field error
            showFieldError('login-password', 'Invalid email or password');
        }
        
        // Reset loading state
        if (window.CommonFunctions) {
            window.CommonFunctions.setButtonLoading(submitBtn, false);
        }
    }, 1500);
}

// Show social login message
function showSocialLoginMessage(provider) {
    if (window.CommonFunctions) {
        window.CommonFunctions.showSuccessNotification(`${provider} login would be implemented in production`);
    }
}

// Show forgot password modal
function showForgotPasswordModal() {
    // Create modal overlay
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: var(--z-modal);
        backdrop-filter: blur(5px);
    `;
    
    // Create modal content
    const modal = document.createElement('div');
    modal.className = 'modal-content';
    modal.innerHTML = `
        <div class="modal-header">
            <h3>Reset Your Password</h3>
            <button class="modal-close">&times;</button>
        </div>
        <div class="modal-body">
            <p>Enter your email address and we'll send you a link to reset your password.</p>
            <div class="form-group">
                <label class="form-label">Email Address</label>
                <input type="email" id="reset-email" class="form-input" placeholder="you@example.com">
                <div class="validation-message" id="reset-email-error"></div>
            </div>
            <button class="submit-btn" id="send-reset-link">Send Reset Link</button>
        </div>
    `;
    
    // Style modal
    modal.style.cssText = `
        background: var(--dark-light);
        border-radius: var(--border-radius-xl);
        padding: var(--spacing-2xl);
        width: 90%;
        max-width: 500px;
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: var(--shadow-xl);
    `;
    
    // Style modal header
    const modalHeader = modal.querySelector('.modal-header');
    modalHeader.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: var(--spacing-xl);
        padding-bottom: var(--spacing-md);
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    `;
    
    // Style modal close button
    const closeBtn = modal.querySelector('.modal-close');
    closeBtn.style.cssText = `
        background: none;
        border: none;
        color: var(--light-color);
        font-size: var(--font-size-2xl);
        cursor: pointer;
        padding: 0;
        width: 30px;
        height: 30px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: var(--border-radius-circle);
        transition: background-color var(--transition-fast);
    `;
    
    closeBtn.addEventListener('mouseenter', function() {
        this.style.background = 'rgba(255, 255, 255, 0.1)';
    });
    
    closeBtn.addEventListener('mouseleave', function() {
        this.style.background = 'none';
    });
    
    // Add close functionality
    closeBtn.addEventListener('click', function() {
        document.body.removeChild(overlay);
    });
    
    overlay.addEventListener('click', function(e) {
        if (e.target === overlay) {
            document.body.removeChild(overlay);
        }
    });
    
    // Add send reset link functionality
    const sendBtn = modal.querySelector('#send-reset-link');
    sendBtn.addEventListener('click', function() {
        const email = modal.querySelector('#reset-email').value.trim();
        const errorElement = modal.querySelector('#reset-email-error');
        
        if (!email) {
            errorElement.textContent = 'Email is required';
            errorElement.className = 'validation-message error';
            return;
        }
        
        if (!isValidEmail(email)) {
            errorElement.textContent = 'Please enter a valid email address';
            errorElement.className = 'validation-message error';
            return;
        }
        
        // Simulate sending reset link
        errorElement.textContent = '';
        sendBtn.textContent = 'Sending...';
        sendBtn.disabled = true;
        
        setTimeout(() => {
            if (window.CommonFunctions) {
                window.CommonFunctions.showSuccessNotification(`Reset link sent to ${email}`);
            }
            document.body.removeChild(overlay);
        }, 1500);
    });
    
    // Add modal to page
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    
    // Focus on email input
    setTimeout(() => {
        modal.querySelector('#reset-email').focus();
    }, 100);
}

// Helper functions
function showFieldError(fieldId, message) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Add error class to input
    field.classList.add('error');
    
    // Show error message
    if (window.CommonFunctions) {
        window.CommonFunctions.showValidationMessage(field, message, 'error');
    }
}

function clearFieldError(fieldId) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Remove error class
    field.classList.remove('error');
    
    // Clear error message
    const errorMessage = field.parentNode.querySelector('.validation-message.error');
    if (errorMessage) {
        errorMessage.remove();
    }
}

function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}